import os
import zipfile
import sys
import re
import time
import random
import json
from tqdm import tqdm
from googletrans import Translator

translator = Translator()
CACHE_FILE = "cache.json"

# Tai cache neu co
if os.path.exists(CACHE_FILE):
    with open(CACHE_FILE, "r", encoding="utf-8") as f:
        cache = json.load(f)
else:
    cache = {}

def save_cache():
    # Luu cache ra file
    with open(CACHE_FILE, "w", encoding="utf-8") as f:
        json.dump(cache, f, ensure_ascii=False, indent=2)

def extract_zip(zip_path, extract_to="en"):
    # Giai nen file zip vao thu muc chi dinh
    with zipfile.ZipFile(zip_path, 'r') as zip_ref:
        zip_ref.extractall(extract_to)
    return extract_to

def should_skip(text):
    """Bo qua cac chuoi khong can dich"""
    # Bo qua neu chuoi chua bien, code, duong dan, HTML tag
    if re.search(r"(\$[A-Za-z0-9_]+|<[^>]+>|{[^}]+}|\[.*\]|http|https|%s|%d|%f)", text):
        return True
    # Bo qua neu chuoi ngan hoac toan ky tu ky hieu
    if len(text.strip()) < 2 or not re.search(r"[A-Za-z]", text):
        return True
    return False

def translate_text(text):
    # Kiem tra cache truoc
    if text in cache:
        return cache[text]
    if should_skip(text):
        cache[text] = text
        return text

    try:
        translated = translator.translate(text, src='en', dest='vi')
        result = translated.text
        cache[text] = result
        save_cache()

        # Tam dung ngau nhien de tranh bi chan
        time.sleep(random.uniform(1.0, 3.0))

        return result
    except Exception as e:
        print("Loi dich:", e)
        return text

def translate_php_file(file_path, output_path):
    # Doc file PHP
    with open(file_path, 'r', encoding='utf-8') as f:
        content = f.read()

    # Bo qua phan comment // va /* */
    content = re.sub(r"//.*?$|/\*.*?\*/", "", content, flags=re.MULTILINE | re.DOTALL)

    # Tim cac chuoi trong dau nhay
    pattern = r"(=>\s*['\"])(.*?)(['\"])"
    matches = re.findall(pattern, content)

    translated = content
    for match in tqdm(matches, desc=f"Dang dich {os.path.basename(file_path)}"):
        prefix, text_to_translate, suffix = match
        if should_skip(text_to_translate):
            continue
        translated_text = translate_text(text_to_translate)
        translated = translated.replace(prefix + text_to_translate + suffix, prefix + translated_text + suffix, 1)

    # Tao thu muc dich neu chua co
    os.makedirs(os.path.dirname(output_path), exist_ok=True)
    with open(output_path, 'w', encoding='utf-8') as f:
        f.write(translated)

def main(zip_path):
    src_folder = extract_zip(zip_path, "en")
    out_folder = "vi"

    # Duyet toan bo file PHP trong thu muc nguon
    for root, dirs, files in os.walk(src_folder):
        for file in files:
            if file.endswith(".php"):
                in_file = os.path.join(root, file)
                rel_path = os.path.relpath(in_file, src_folder)
                out_file = os.path.join(out_folder, rel_path)
                translate_php_file(in_file, out_file)
                print(f"✅ Da dich xong: {rel_path}")

if __name__ == "__main__":
    if len(sys.argv) < 2:
        print("Cach su dung: python translate_lang_folder.py en.zip")
    else:
        main(sys.argv[1])
